<?php

namespace App\Http\Controllers\Admin;

use App\Classes\GeniusMailer;
use App\Http\Controllers\Controller;
use App\Models\Currency;
use App\Models\Deposit;
use App\Models\Generalsetting;
use App\Models\PaymentGateway;
use App\Models\Transaction;
use App\Models\User;
use Datatables;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\Log;
use PDF;
use App\Exports\StatementDepositExport;
use Maatwebsite\Excel\Facades\Excel;
use DB;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;


class DepositController extends Controller
{
    public function datatables()
    {
        $data = Deposit::with(['user', 'gateway', 'lastHistory']);

        if (request()->filled('from_date')) {
            $data->whereDate('created_at', '>=', request('from_date'));
        }

        if (request()->filled('to_date')) {
            $data->whereDate('created_at', '<=', request('to_date'));
        }

        return Datatables::of($data)
            ->editColumn('last_history', function ($data) {
                if ($data->lastHistory) {
                    $name = e($data->lastHistory->admin->name);
                    $time = $data->lastHistory->created_at;
                    return "{$name}<br><small>{$time}</small>";
                }
            })
            ->editColumn('created_at', function ($row) {
                return date('d M Y', strtotime($row->created_at));
            })
            ->filterColumn('user.name', function (Builder $query, $keyword) {
                $query->whereRelation('user', 'name', 'like', "%$keyword%");
            })
            ->filterColumn('customer_name', function ($query, $keyword) {
                $query->whereHas('user', function ($q) use ($keyword) {
                    $q->where('name', 'like', "%$keyword%");
                });
            })
            ->filterColumn('customer_email', function ($query, $keyword) {
                $query->whereHas('user', function ($q) use ($keyword) {
                    $q->where('email', 'like', "%$keyword%");
                });
            })
            ->editColumn('created_at', function (Deposit $row) {
                return date('d-M-Y, H:i', strtotime($row->created_at));
            })
            ->addColumn('customer_name', function (Deposit $row) {
                return $row->user->name;
            })
            ->addColumn('customer_email', function (Deposit $row) {
                return $row->user->email;
            })
            ->editColumn('method', function (Deposit $row) {
                if ($row->method == 'Manual') {
                    return $row->gateway ? ucwords($row->gateway->title ?? '') . ' (Manual)' : 'Manual';
                }
                return ucwords($row->gateway->name ?? $row->method);
            })
            ->editColumn('amount', function (Deposit $row) {
                $gs = Generalsetting::find(1);
                $defaultCurrency = Currency::where('is_default', 1)->first();
                return $defaultCurrency->sign . round($row->amount * $defaultCurrency->value, 2);
            })
            ->editColumn('status', function (Deposit $row) {
                [$status_label, $status_sign] = match ($row->status) {
                    "processing" => [__("Processing"), "info"],
                    "complete" => [__("Completed"), "success"],
                    "pending" => [__("Pending"), "warning"],
                    "onhold" => [__("On Hold"), "danger"],
                    default => [__("Canceled"), "danger"],
                };

                $html = <<<'HTML'
                    <div class="btn-group mb-1">
                        <button type="button" class="btn btn-{{$status_sign}} btn-sm btn-rounded @if(!$disabled) dropdown-toggle @endif"
                                @if(!$disabled) data-toggle="dropdown" aria-haspopup="true" aria-expanded="false" @endif>
                          {{ $status_label }}
                        </button>
                        @if(!$disabled)
                        <div class="dropdown-menu" x-placement="bottom-start">
                          <a href="#statusModal" data-toggle="modal" class="dropdown-item" data-href="{{route('admin.deposits.status', ['id1' => $id, 'id2' => 'pending'])}}">{{__("Pending")}}</a>
                          <a href="#statusModal" data-toggle="modal" class="dropdown-item" data-href="{{route('admin.deposits.status', ['id1' => $id, 'id2' => 'complete'])}}">{{__("Completed")}}</a>
                          <a href="#statusModal" data-toggle="modal" class="dropdown-item" data-href="{{route('admin.deposits.status', ['id1' => $id, 'id2' => 'processing'])}}">{{__("Processing")}}</a>
                          <a href="#statusModal" data-toggle="modal" class="dropdown-item" data-href="{{route('admin.deposits.status', ['id1' => $id, 'id2' => 'canceled'])}}">{{__("Cancel")}}</a>
                          <a href="#statusModal" data-toggle="modal" class="dropdown-item" data-href="{{route('admin.deposits.status', ['id1' => $id, 'id2' => 'onhold'])}}">{{__("Hold")}}</a>
                        </div>
                        @endif
                    </div>
                HTML;
                return Blade::render($html, array_merge($row->toArray(), [
                    'disabled' => in_array($row->status, ['']),
                    // 'disabled' => in_array($row->status, ['complete', 'canceled']),
                    'status_label' => $status_label,
                    'status_sign' => $status_sign,
                ]));
            })
            ->addColumn('view', '<a href="#viewModal" data-url="{{route(\'admin.deposits.show\',$id)}}" class="btn btn-primary btn-sm view-deposit" data-toggle="modal">View</a>')
            ->rawColumns(['last_history', 'created_at', 'customer_name', 'customer_email', 'amount', 'status', 'view'])
            ->toJson();
    }

    public function index()
    {
        $data['pending'] = Deposit::whereStatus('pending')->count();
        $data['complete'] = Deposit::whereStatus('complete')->count();
        $data['cancelled'] = Deposit::whereStatus('canceled')->count();
        $data['processing'] = Deposit::whereStatus('processing')->count();
        $data['onhold'] = Deposit::whereStatus('onhold')->count();
        // return Deposit::get();
        return view('admin.deposit.index', $data);
    }

    public function status_old($id1, $id2)
    {
        $data = Deposit::findOrFail($id1);

        if ($data->status == $id2) {
            $msg = "Deposits already $id2";
            return response()->json($msg);
        }

        $user = User::findOrFail($data->user_id);


        if ($id2 == 'complete') {
            $user->balance += $data->amount;
            $user->save();

            $fund = PaymentGateway::query()->whereId($data->gateway_id)->first();
            $fund->fund_available += $data->amount * $fund->receiving_rate;
            $fund->save();

            $trans = new Transaction();
            $trans->email = $user->email;
            $trans->amount = $data->amount;
            $trans->type = "Deposit";
            $trans->profit = "plus";
            $trans->txnid = $data->deposit_number;
            $trans->user_id = $user->id;
            $trans->user_available_balance = $user->balance;
            $trans->save();
        }

        $data->update(['status' => $id2]);

        try {
            $gs = Generalsetting::findOrFail(1);
            if ($gs->is_smtp == 1) {
                $data = [
                    'to' => $user->email,
                    'type' => "Deposti",
                    'cname' => $user->name,
                    'oamount' => $data->amount,
                    'aname' => "",
                    'aemail' => "",
                    'wtitle' => "",
                ];

                $mailer = new GeniusMailer();
                $mailer->sendAutoMail($data);
            } else {
                $to = $user->email;
                $subject = " You have deposited successfully.";
                $msg = "Hello " . $user->name . "!\nYou have invested successfully.\nThank you.";
                $headers = "From: " . $gs->from_name . "<" . $gs->from_email . ">";
                mail($to, $subject, $msg, $headers);
            }
        } catch (\Exception $e) {
            Log::error($e);
        }

        $msg = 'Data Updated Successfully.';
        return response()->json($msg);
    }


    public function status($id1, $id2)
    {
        $data = Deposit::findOrFail($id1);

        if ($data->status === $id2) {
            return response()->json("Deposits already $id2");
        }

        $user = User::findOrFail($data->user_id);
        $authUser =  Auth::guard('admin')->user(); // Logged-in admin or user

        // Check role-based permission
        if (
            !$authUser->IsSuper() &&
            ($data->status == 'complete' || $data->status == 'canceled')

        ) {

            $now = Carbon::now();
              $lastUpdate = Carbon::parse($data->status_updated_at);

              $diff = $lastUpdate->diffInMinutes($now, false);
            if ($diff > 60) {
              //   dd("You do not have permission to change this status.");
                return response()->json( 'You can no longer update this status. 60-minute window has passed.'
                , 200);
            }
    
         
        }

        
        // Allow superadmin or 'user' (within 30 minutes)
        if ($id2 === 'complete' && $data->status !== 'complete') {
            //   return $data;
            if ($data->is_fund_add == 0) {
                $fund = PaymentGateway::query()->whereId($data->gateway_id)->first();
                $fund->fund_available += $data->amount * $fund->receiving_rate;
                // $fund->fund_available +=  $data->charge * $fund->receiving_rate;
                $fund->save();
                $data->is_fund_add = 1;
                $data->save();
            }

            if ($data->is_amount_add == 0) {
                $user->balance += $data->amount - $data->charge ;
                $user->save();

                $trans = new Transaction();
                $trans->email = $user->email;
                $trans->amount = $data->amount;
                $trans->type = "Deposit";
                $trans->profit = "plus";
                $trans->txnid = $data->deposit_number;
                $trans->charge = $data->charge;
                $trans->user_id = $user->id;
                $trans->user_available_balance = $user->balance;
                $trans->save();


                $data->is_amount_add = 1;
                $data->save();
            }
        }

        if ($id2 === 'canceled'  && $data->status == 'complete') {


            if ($data->is_fund_add == 1) {
                $fund = PaymentGateway::query()->whereId($data->gateway_id)->first();
                $fund->fund_available -= $data->amount * $fund->receiving_rate;
                $fund->save();
                $data->is_fund_add = 0;
                $data->save();
            }

            if ($data->is_amount_add == 1) {
                $user->balance -= ($data->amount - $data->charge);
                $user->save();
                $trans = new Transaction();
                $trans->email = $user->email;
                $trans->amount = $data->amount;
                $trans->type = "Deposit";
                $trans->profit = "minus";
                $trans->txnid = $data->deposit_number;
                $trans->charge = $data->charge;
                $trans->user_id = $user->id;
                $trans->user_available_balance = $user->balance;
                $trans->save();
                $data->is_amount_add = 0;
                $data->save();
            }
        }
        


        if ($id2 === 'canceled'  && ($data->status == 'onhold' || $data->status == 'processing' || $data->status == 'pending')) {
            // $user->balance -= ($data->amount - $data->charge);
            // $user->save();

            if ($data->is_fund_add == 1) {
                $fund = PaymentGateway::query()->whereId($data->gateway_id)->first();
                $fund->fund_available -= $data->amount * $fund->receiving_rate;
                $fund->save();
                $data->is_fund_add = 0;
                $data->save();
            }

            if ($data->is_amount_add == 1) {
                $user->balance -= ($data->amount - $data->charge);
                $user->save();
                $trans = new Transaction();
                $trans->email = $user->email;
                $trans->amount = $data->amount;
                $trans->type = "Deposit";
                $trans->profit = "minus";
                $trans->txnid = $data->deposit_number;
                $trans->charge = $data->charge;
                $trans->user_id = $user->id;
                $trans->user_available_balance = $user->balance;
                $trans->save();
                $data->is_amount_add = 0;
                $data->save();
            }
        }


        if (($id2 === 'onhold' || $id2 === 'processing' || $id2 == "pending")) {
            dd($data);
            $fund = PaymentGateway::query()->whereId($data->gateway_id)->first();
            $fund->fund_available -= $data->amount * $fund->receiving_rate;
            $fund->save();
            $data->is_fund_add = 0;
            $data->save();
        }

        // Update deposit status
        statusHistory($data->id, 'deposit', $data->status, $id2);
        $data->status = $id2;
        $data->status_updated_at = now(); // Optional: track status update separately
        $data->save();

        $this->MailStatusCompltete($user);
        return response()->json('Data Updated Successfully.');
    }


    public function show(Deposit $id)
    {
        $id->load(['histories']);
        $depositCurrency = Currency::find($id->currency_id);
        $gateway = PaymentGateway::find($id->gateway_id);
        $defaultCurrency = Currency::where('is_default', 1)->first();

        return view('admin.deposit.show', [
            'depositCurrency' => $depositCurrency,
            'defCurrency' => $defaultCurrency,
            'deposit' => $id,
            'gateway' => $gateway
        ]);
    }

    public function exportPdf()
    {
        $gs = Generalsetting::first();
        $data = [
            'title' => $gs->title,
            'date' => date('Y-m-d'),
            'from' => request('from'),
            'to' => request('to'),
        ];

        $query = Deposit::with('user', 'lastHistory')->orderBy('id', 'desc');

        if (request()->filled(['from', 'to'])) {
            $query->whereBetween(DB::raw('DATE(created_at)'), [request('from'), request('to')]);
        }

        $data['deposits'] = $query->get();

        $pdf = PDF::loadView('admin.deposit.statement-pdf', $data);

        return $pdf->stream(uniqid('deposits-') . '.pdf');
    }

    public function exportExcel()
    {
        $gs = Generalsetting::first();
        $data = [
            'title' => $gs->title,
            'date' => date('Y-m-d'),
            'from' => request('from'),
            'to' => request('to'),
        ];

        $query = Deposit::with('user', 'lastHistory')->orderBy('id', 'desc');
        if (request()->filled(['from', 'to'])) {
            $query->whereBetween(DB::raw('DATE(created_at)'), [request('from'), request('to')]);
        }
        $data['deposits'] = $query->get();

        return Excel::download(new StatementDepositExport($data), uniqid('deposits-') . '.xlsx', \Maatwebsite\Excel\Excel::XLSX);
    }


    public function MailStatusCompltete($user)
    {
        try {
            $gs = Generalsetting::findOrFail(1);
            if ($gs->is_smtp == 1) {
                $emailData = [
                    'to' => $user->email,
                    'type' => "Deposit",
                    'cname' => $user->name,
                    'oamount' => $user->amount,
                    'aname' => "",
                    'aemail' => "",
                    'wtitle' => "",
                ];
                $mailer = new GeniusMailer();
                $mailer->sendAutoMail($emailData);
            } else {
                $to = $user->email;
                $subject = "You have deposited successfully.";
                $msg = "Hello " . $user->name . "!\nYou have invested successfully.\nThank you.";
                $headers = "From: " . $gs->from_name . "<" . $gs->from_email . ">";
                mail($to, $subject, $msg, $headers);
            }
        } catch (\Exception $e) {
            Log::error($e);
        }
    }
}
